<?php
/**
 * generator.php (self-hosted final)
 *
 * Erzeugt QR-Codes lokal als PNG oder SVG mit einstellbaren Parametern.
 * Keine externen Dienste, keine Abhängigkeiten, kein Composer nötig.
 */

require_once __DIR__.'/lib/autoload.php';

ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);
error_reporting(E_ALL);

use chillerlan\QRCode\QRCode;
use chillerlan\QRCode\QROptions;
use chillerlan\QRCode\Output\QRGdImagePNG;
use chillerlan\QRCode\Output\QRMarkupSVG;
use chillerlan\QRCode\Common\EccLevel;
use chillerlan\QRCode\Data\QRMatrix;

$data     = $_GET['data']     ?? 'https://b-prisma.de';
$format   = $_GET['format']   ?? 'png';
$filename = $_GET['filename'] ?? 'qrcode.' . $format;
$bg       = isset($_GET['bg']) ? $_GET['bg'] : 'transparent';
$fg       = isset($_GET['fg']) ? $_GET['fg'] : '000000';
$size     = (int)($_GET['size'] ?? 300);

// Hilfsfunktion: Hex-Farbe zu RGB-Array konvertieren
function hexToRgb($hex) {
    $hex = ltrim($hex, '#');
    if (strlen($hex) === 3) {
        $hex = $hex[0] . $hex[0] . $hex[1] . $hex[1] . $hex[2] . $hex[2];
    }
    return [
        hexdec(substr($hex, 0, 2)),
        hexdec(substr($hex, 2, 2)),
        hexdec(substr($hex, 4, 2))
    ];
}

// Zuerst QR-Code-Matrix erstellen, um Modulanzahl zu ermitteln
// Wir müssen die Daten zuerst hinzufügen, bevor wir getQRMatrix() aufrufen können
$tempOptions = new QROptions([
    'outputInterface' => $format === 'svg' ? QRMarkupSVG::class : QRGdImagePNG::class,
    'eccLevel'        => EccLevel::L,
    'scale'           => 1, // Temporärer Wert
]);

$tempQr = new QRCode($tempOptions);
// Daten hinzufügen - render() fügt die Segmente hinzu, aber rendert auch
// Wir verwenden einen Workaround: render() aufrufen, aber die Ausgabe ignorieren
ob_start();
$tempQr->render($data);
ob_end_clean();
// Jetzt können wir die Matrix abrufen
$matrix = $tempQr->getQRMatrix();
$moduleCount = $matrix->getSize();

// Scale basierend auf gewünschter Größe berechnen
$scale = max(1, (int)($size / $moduleCount));

// Farben vorbereiten - unterschiedlich für PNG (RGB-Array) und SVG (CSS-String)
if ($format === 'svg') {
    // SVG benötigt CSS-Farbstrings
    $bgColor = $bg === 'transparent' ? 'none' : '#' . ltrim($bg, '#');
    $fgColor = '#' . ltrim($fg, '#');
    
    $options = new QROptions([
        'outputInterface'         => QRMarkupSVG::class,
        'eccLevel'                => EccLevel::L,
        'scale'                   => $scale,
        'outputBase64'            => false,
        'bgColor'                 => $bgColor,
        'moduleValues'            => [
            // Dunkle Module (Vordergrund)
            QRMatrix::M_DARKMODULE     => $fgColor,
            QRMatrix::M_DATA_DARK      => $fgColor,
            QRMatrix::M_FINDER_DARK    => $fgColor,
            QRMatrix::M_SEPARATOR_DARK => $fgColor,
            QRMatrix::M_ALIGNMENT_DARK => $fgColor,
            QRMatrix::M_TIMING_DARK    => $fgColor,
            QRMatrix::M_FORMAT_DARK    => $fgColor,
            QRMatrix::M_VERSION_DARK   => $fgColor,
            QRMatrix::M_QUIETZONE_DARK => $fgColor,
            QRMatrix::M_LOGO_DARK      => $fgColor,
            QRMatrix::M_FINDER_DOT     => $fgColor,
            // Helle Module (Hintergrund) - WICHTIG: Diese müssen auch gesetzt werden!
            QRMatrix::M_DATA             => $bgColor,
            QRMatrix::M_FINDER           => $bgColor,
            QRMatrix::M_SEPARATOR        => $bgColor,
            QRMatrix::M_ALIGNMENT        => $bgColor,
            QRMatrix::M_TIMING           => $bgColor,
            QRMatrix::M_FORMAT           => $bgColor,
            QRMatrix::M_VERSION          => $bgColor,
            QRMatrix::M_QUIETZONE        => $bgColor,
            QRMatrix::M_DARKMODULE_LIGHT => $bgColor,
            QRMatrix::M_FINDER_DOT_LIGHT => $bgColor,
            QRMatrix::M_LOGO             => $bgColor,
            QRMatrix::M_NULL             => $bgColor,
        ],
    ]);
} else {
    // PNG benötigt RGB-Arrays
    if ($bg === 'transparent' || $bg === '') {
        $bgRgb = [255, 255, 255];
        $imageTransparent = true;
    } else {
        // Stelle sicher, dass bg nicht leer ist
        if (empty($bg)) {
            $bgRgb = [255, 255, 255];
        } else {
            $bgRgb = hexToRgb($bg);
        }
        $imageTransparent = false;
    }
    $fgRgb = hexToRgb($fg);
    
    // Stelle sicher, dass bgRgb ein gültiges Array ist
    if (!is_array($bgRgb) || count($bgRgb) !== 3) {
        $bgRgb = [255, 255, 255];
    }
    
    // Stelle sicher, dass bgRgb numerische Werte hat
    $bgRgb = [
        (int)$bgRgb[0],
        (int)$bgRgb[1],
        (int)$bgRgb[2]
    ];
    
    $options = new QROptions([
        'outputInterface'         => QRGdImagePNG::class,
        'eccLevel'                => EccLevel::L,
        'scale'                   => $scale,
        'outputBase64'            => false,
        'imageTransparent'        => $imageTransparent,
        'transparencyColor'       => $imageTransparent ? $bgRgb : null,
        'bgColor'                 => $bgRgb,  // Explizit als Array setzen
        'moduleValues'            => [
            // Dunkle Module (Vordergrund)
            QRMatrix::M_DARKMODULE     => $fgRgb,
            QRMatrix::M_DATA_DARK      => $fgRgb,
            QRMatrix::M_FINDER_DARK    => $fgRgb,
            QRMatrix::M_SEPARATOR_DARK => $fgRgb,
            QRMatrix::M_ALIGNMENT_DARK => $fgRgb,
            QRMatrix::M_TIMING_DARK    => $fgRgb,
            QRMatrix::M_FORMAT_DARK    => $fgRgb,
            QRMatrix::M_VERSION_DARK   => $fgRgb,
            QRMatrix::M_QUIETZONE_DARK => $fgRgb,
            QRMatrix::M_LOGO_DARK      => $fgRgb,
            QRMatrix::M_FINDER_DOT     => $fgRgb,
            // Helle Module (Hintergrund) - WICHTIG: Diese müssen auch gesetzt werden!
            QRMatrix::M_DATA             => $bgRgb,
            QRMatrix::M_FINDER           => $bgRgb,
            QRMatrix::M_SEPARATOR        => $bgRgb,
            QRMatrix::M_ALIGNMENT        => $bgRgb,
            QRMatrix::M_TIMING           => $bgRgb,
            QRMatrix::M_FORMAT           => $bgRgb,
            QRMatrix::M_VERSION          => $bgRgb,
            QRMatrix::M_QUIETZONE        => $bgRgb,
            QRMatrix::M_DARKMODULE_LIGHT => $bgRgb,
            QRMatrix::M_FINDER_DOT_LIGHT => $bgRgb,
            QRMatrix::M_LOGO             => $bgRgb,
            QRMatrix::M_NULL             => $bgRgb,
        ],
    ]);
}

$qr = new QRCode($options);

// Header für Dateidownload
header('Content-Type: ' . ($format === 'svg' ? 'image/svg+xml' : 'image/png'));
header('Content-Disposition: attachment; filename="' . basename($filename) . '"');

// Ausgabe
$output = $qr->render($data);

// Für SVG: width und height Attribute hinzufügen, falls nicht vorhanden
if ($format === 'svg') {
    // Prüfe, ob width/height bereits vorhanden sind
    if (strpos($output, 'width=') === false && strpos($output, 'height=') === false) {
        // Füge width und height zum SVG-Tag hinzu
        $output = preg_replace(
            '/<svg([^>]*)>/',
            '<svg$1 width="' . $size . '" height="' . $size . '">',
            $output,
            1
        );
    }
}

echo $output;