# -*- coding: utf-8 -*-
"""
GDLnucleus Light - Kostenlose Version für GSM <-> HSF Konvertierung
Handhabt die Umwandlung von ArchiCAD GDL-Bibliothekselementen zwischen GSM und HSF Formaten.
Unterstützt: GSM2HSF und HSF2GSM Konvertierungen mit Backup-Funktionen.
"""
import datetime
import os
import shutil
import glob
import sys
import platform
import logging
from subprocess import call
from os.path import join as pjoin, isdir
import json

# Globale Konfigurationsvariable
C = None

# Konstanten für Ordnerstrukturen - Light Version (nur GSM2HSF und HSF2GSM)
# FOLDER_GSM_BASE: Quell-GSM-Dateien (01_gsms)
# FOLDER_HSF: HSF-Quelldateien (02_hsf)
# FOLDER_BACKUPS: Arbeits- und Backup-Verzeichnis (03_backups)
# FOLDER_LOGS: Log-Dateien (04_logs)
FOLDER_GSM_BASE = '01_gsms'
FOLDER_HSF = '02_hsf'
FOLDER_BACKUPS = '03_backups'
FOLDER_LOGS = '04_logs'

# Diese werden nach dem Laden der Konfiguration initialisiert
GSM_LIBRARY_NAME = None
FOLDER_GSM_SOURCE = None
FOLDERS_GSMS = None
project_path = None
ziel = None
quelle = None
logger = None

# Debugging-Option
PRINT_TIMES = 1


def setup_logging(projectpath):
    """
    Initialisiert das Logging-System.
    
    Args:
        projectpath: Pfad zum Projektverzeichnis
        
    Returns:
        Logger-Instanz
    """
    global logger
    
    log_dir = pjoin(projectpath, FOLDER_LOGS)
    if not os.path.exists(log_dir):
        os.makedirs(log_dir)
    
    log_file = pjoin(log_dir, f'gdlconverter_{datetime.datetime.now().strftime("%Y%m%d")}.log')
    
    # Logger konfigurieren
    logger = logging.getLogger('gdlconverter')
    logger.setLevel(logging.DEBUG)
    
    # Handler für Datei (DEBUG und höher)
    file_handler = logging.FileHandler(log_file, encoding='utf-8')
    file_handler.setLevel(logging.DEBUG)
    file_format = logging.Formatter(
        '%(asctime)s - %(levelname)s - %(message)s',
        datefmt='%Y-%m-%d %H:%M:%S'
    )
    file_handler.setFormatter(file_format)
    
    # Handler für Konsole (INFO und höher)
    console_handler = logging.StreamHandler(sys.stdout)
    console_handler.setLevel(logging.INFO)
    console_format = logging.Formatter('%(levelname)s - %(message)s')
    console_handler.setFormatter(console_format)
    
    # Handler hinzufügen
    logger.addHandler(file_handler)
    logger.addHandler(console_handler)
    
    return logger


def detect_library_name(projectpath):
    """
    Ermittelt den Bibliotheksnamen automatisch aus dem ersten Ordnernamen unter 01_gsms/.
    Ignoriert Ordner mit "-Demo" Suffix.
    
    Args:
        projectpath: Pfad zum Projektverzeichnis
        
    Returns:
        Bibliotheksname oder None wenn nicht gefunden
    """
    gsm_base_path = pjoin(projectpath, FOLDER_GSM_BASE)
    
    if not os.path.exists(gsm_base_path):
        return None
    
    # Alle Ordner unter 01_gsms/ auflisten
    folders = [f for f in os.listdir(gsm_base_path) 
               if isdir(pjoin(gsm_base_path, f)) and not f.startswith('.')]
    
    # Sortieren und den ersten Ordner ohne "-Demo" Suffix nehmen
    folders.sort()
    for folder in folders:
        if not folder.endswith('-Demo'):
            return folder
    
    # Falls nur Demo-Ordner vorhanden, den ersten nehmen und "-Demo" entfernen
    if folders:
        return folders[0].replace('-Demo', '')
    
    return None


def gsm_backups(file2backup, filename):
    """
    Erstellt Backups einer GSM-Datei in konfigurierten Backup-Verzeichnissen.
    
    Args:
        file2backup: Pfad zur zu sichernden Datei
        filename: Dateiname ohne Pfad und Erweiterung
    """
    if 'gsmbackupdirectories' not in C:
        error_msg = 'No GSM backup directories ("gsmbackupdirectories") defined in gdlconfig.json! If no backups are needed, use this:\n"gsmbackupdirectories: []'
        if logger:
            logger.error(error_msg)
        print(error_msg)
        sys.exit(1)
        
    for backupdirectory in C['gsmbackupdirectories']:
        if 'directory' not in backupdirectory or 'filename' not in backupdirectory:
            error_msg = 'GSM backup directory configuration ("gsmbackupdirectories") defined in gdlconfig.json is missing "directory" or "filename" key!'
            if logger:
                logger.warning(error_msg)
            print(error_msg)
            continue
            
        backupfolder = backupdirectory['directory'].replace('{$projectpath}', C['projectpath'])
        backupfilename = backupdirectory['filename'].replace('{$filename}', filename)
        
        # Ersetze Zeitplatzhalter falls vorhanden
        if 'strftime' in backupdirectory:
            backupstrftime = datetime.datetime.now().strftime(backupdirectory['strftime'])
            backupfolder = backupfolder.replace('{$strftime}', backupstrftime)
            backupfilename = backupfilename.replace('{$strftime}', backupstrftime)
            
        # Erstelle Backup-Verzeichnis falls nötig
        if not os.path.exists(backupfolder):
            os.makedirs(backupfolder)
            
        # Kopiere die Datei ins Backup-Verzeichnis
        try:
            shutil.copy(file2backup, pjoin(backupfolder, backupfilename))
            if logger:
                logger.debug(f'Backup erstellt: {pjoin(backupfolder, backupfilename)}')
        except Exception as e:
            error_msg = f'Fehler beim Erstellen des Backups: {e}'
            if logger:
                logger.error(error_msg)
            print(error_msg)


def format_timedelta(td):
    """
    Formatiert ein timedelta-Objekt als lesbare Zeichenkette.
    
    Args:
        td: Ein datetime.timedelta Objekt
        
    Returns:
        Formatierte Zeit in Sekunden
    """
    return str(td.seconds+td.microseconds/1000000.0)+'s'


def get_lpxmlconverter_path():
    """
    Ermittelt den Pfad zum LP_XMLConverter basierend auf Plattform und Konfiguration.
    
    Returns:
        Pfad zum LP_XMLConverter oder None wenn nicht gefunden
    """
    # Zuerst prüfen ob direkt in Konfiguration definiert
    if 'lpxmlconverter' in C and C['lpxmlconverter']:
        converter_path = C['lpxmlconverter']
        if os.path.exists(converter_path) and os.access(converter_path, os.X_OK):
            return converter_path
    
    # Plattform-spezifische Pfadauswahl
    system = platform.system().lower()
    if 'platforms' in C and system in C['platforms']:
        platform_config = C['platforms'][system]
        if 'lpxmlconverter' in platform_config:
            converter_path = platform_config['lpxmlconverter']
            if os.path.exists(converter_path) and os.access(converter_path, os.X_OK):
                return converter_path
    
    return None


def lpxmlconverter(mode, *args):
    """
    Führt den lpxmlconverter mit den angegebenen Parametern aus.
    
    Args:
        mode: Konvertierungsmodus (l2hsf, hsf2l, etc.)
        *args: Argumente für den Konverter (Pfade, Optionen)
        
    Returns:
        Exit-Code des Konverter-Prozesses
    """
    start = datetime.datetime.now()
    args = list(args)
    
    # Pfad zum Konverter ermitteln
    converter_path = get_lpxmlconverter_path()
    if not converter_path:
        error_msg = 'Error: lpxmlconverter path not found. Please check gdlconfig.json'
        if logger:
            logger.error(error_msg)
            logger.error(f'Current platform: {platform.system()}')
            logger.error('Checked paths:')
            if 'lpxmlconverter' in C:
                logger.error(f'  - {C["lpxmlconverter"]}')
            if 'platforms' in C:
                for plat, config in C['platforms'].items():
                    if 'lpxmlconverter' in config:
                        logger.error(f'  - {plat}: {config["lpxmlconverter"]}')
        print(error_msg)
        print(f'  Current platform: {platform.system()}')
        print(f'  Checked paths:')
        if 'lpxmlconverter' in C:
            print(f'    - {C["lpxmlconverter"]}')
        if 'platforms' in C:
            for plat, config in C['platforms'].items():
                if 'lpxmlconverter' in config:
                    print(f'    - {plat}: {config["lpxmlconverter"]}')
        return 1
        
    args = [converter_path, mode] + args
    if logger:
        logger.info(f'Execute: {" ".join(args)}')
    print('Execute', args)
    exitcode = call(args)
    if PRINT_TIMES:
        time_str = format_timedelta(datetime.datetime.now()-start)
        print('Execute time:', time_str)
        if logger:
            logger.debug(f'Execute time: {time_str}')
    return exitcode


def gsm2hsf(gsmfile):
    """
    Konvertiert eine GSM-Datei ins HSF-Format.
    
    Args:
        gsmfile: Pfad zur GSM-Datei
    """
    filename = os.path.splitext(os.path.basename(gsmfile))[0]
    infolder = pjoin(C['projectpath'], FOLDER_BACKUPS, filename)
    sourcefolder = pjoin(C['projectpath'], FOLDER_HSF, filename)
    
    # GSM-Datei kopieren
    if not os.path.exists(infolder):
        os.makedirs(infolder)
    shutil.copy(gsmfile, infolder)
    
    # HSF-Zielordner erstellen falls nötig
    if not os.path.exists(sourcefolder):
        os.makedirs(sourcefolder)
    
    # Konvertierung durchführen - LP_XMLConverter erstellt HSF-Dateien direkt
    # l2hsf "binaryFolderPath" "hsfFolderPath"
    exitcode = lpxmlconverter('l2hsf',
            pjoin(C['projectpath'], FOLDER_BACKUPS, filename),
            pjoin(C['projectpath'], FOLDER_HSF, filename))
    
    return exitcode


def hsf2gsm(gsmfile):
    """
    Konvertiert HSF-Quelldateien zurück in eine GSM-Datei.
    LP_XMLConverter erstellt immer Unterordner - wir verschieben die GSM direkt.
    
    Args:
        gsmfile: Pfad zur Ziel-GSM-Datei
    """
    filename = os.path.splitext(os.path.basename(gsmfile))[0]
    hsf_base_folder = pjoin(C['projectpath'], FOLDER_HSF)
    gsm_target_folder = os.path.dirname(gsmfile)
    
    # Prüfen ob HSF-Ordner für diese Datei existiert
    hsf_file_folder = pjoin(hsf_base_folder, filename)
    if not os.path.exists(hsf_file_folder):
        error_msg = f'HSF folder not found: {hsf_file_folder}'
        if logger:
            logger.error(error_msg)
        print(error_msg)
        return 1
    
    # Backup der bestehenden GSM-Datei erstellen
    if os.path.exists(gsmfile):
        gsm_backups(gsmfile, filename)
    
    # In temporären Ordner konvertieren (LP_XMLConverter erstellt Unterordner)
    temp_output_folder = pjoin(C['projectpath'], FOLDER_BACKUPS, 'temp_hsf2l')
    if os.path.exists(temp_output_folder):
        shutil.rmtree(temp_output_folder)
    os.makedirs(temp_output_folder)
    
    # Konvertierung durchführen
    exitcode = lpxmlconverter('hsf2l',
            hsf_base_folder,
            temp_output_folder)
    
    if exitcode == 0:
        # LP_XMLConverter erstellt: temp_hsf2l/Flexi-Text-Label/Flexi-Text-Label.gsm
        # Wir wollen: 01_gsms/gdl-nucleus-light/Flexi-Text-Label.gsm
        created_subfolder = pjoin(temp_output_folder, filename)
        created_gsm = pjoin(created_subfolder, filename + '.gsm')
        
        if os.path.exists(created_gsm):
            # Verschiebe die GSM direkt in den Zielordner
            shutil.move(created_gsm, gsmfile)
            if logger:
                logger.info(f'Successfully converted HSF to GSM: {gsmfile}')
        else:
            error_msg = f'GSM file not found at: {created_gsm}'
            if logger:
                logger.error(error_msg)
            print(error_msg)
            exitcode = 1
        
        # Temporären Ordner aufräumen
        if os.path.exists(temp_output_folder):
            shutil.rmtree(temp_output_folder)
    else:
        error_msg = 'HSF2GSM conversion failed!'
        if logger:
            logger.error(error_msg)
        print(error_msg)
        
        # Temporären Ordner aufräumen
        if os.path.exists(temp_output_folder):
            shutil.rmtree(temp_output_folder)
    
    return exitcode




if __name__=='__main__':
    """
    Hauptprogramm - Verarbeitet Befehlszeilenargumente und führt die entsprechenden Aktionen aus.
    """
    if PRINT_TIMES: print('parameters', sys.argv[1:])
    
    # Parameter prüfen
    if len(sys.argv)<4:
        print('not enough parameters! gdlconverter.py mode foldername projectpath')
        sys.exit(1)
        
    mode = sys.argv[1]
    foldername = sys.argv[2]
    projectpath = sys.argv[3]
    
    # Logging initialisieren (muss früh erfolgen)
    setup_logging(projectpath)
    if logger:
        logger.info(f'Starting gdlconverter.py with mode: {mode}, folder: {foldername}')
    
    # Konfiguration laden
    config_path = pjoin(projectpath, 'gdlconfig.json')
    try:
        with open(config_path, 'r', encoding='utf-8') as f:
            C = json.loads(f.read())
        if logger:
            logger.debug(f'Configuration loaded from {config_path}')
    except FileNotFoundError:
        error_msg = f"Error: Configuration file {config_path} not found"
        if logger:
            logger.error(error_msg)
        print(error_msg)
        sys.exit(1)
    except json.JSONDecodeError as e:
        error_msg = f"Error parsing JSON configuration: {str(e)}"
        if logger:
            logger.error(error_msg)
        print(error_msg)
        sys.exit(1)
        
    C['projectpath'] = projectpath
    
    # Bibliotheksname automatisch aus dem ersten Ordnernamen unter 01_gsms/ ermitteln
    detected_name = detect_library_name(projectpath)
    if not detected_name:
        error_msg = f"Error: No library folder found in {pjoin(projectpath, FOLDER_GSM_BASE)}"
        if logger:
            logger.error(error_msg)
        print(error_msg)
        sys.exit(1)
    
    GSM_LIBRARY_NAME = detected_name
    
    if logger:
        logger.info(f'Detected library name: {GSM_LIBRARY_NAME}')
    
    # Globale Pfade initialisieren
    FOLDER_GSM_SOURCE = f'{FOLDER_GSM_BASE}/{GSM_LIBRARY_NAME}'
    # Auch Demo-Ordner berücksichtigen
    FOLDERS_GSMS = [
        FOLDER_GSM_SOURCE, 
        f'{FOLDER_GSM_SOURCE}/Makros',
        f'{FOLDER_GSM_BASE}/{GSM_LIBRARY_NAME}-Demo',
        f'{FOLDER_GSM_BASE}/{GSM_LIBRARY_NAME}-Demo/Makros'
    ]
    project_path = projectpath
    
    # Debug-Ausgabe konfigurieren
    if 'printtimes' not in C or not C['printtimes']:
        PRINT_TIMES = 0
    
    # Alle GSM-Dateien in den konfigurierten Ordnern verarbeiten
    for folder in FOLDERS_GSMS:
        folder_path = pjoin(projectpath, folder)
        if not os.path.exists(folder_path):
            continue
            
        gsmfiles = glob.glob(pjoin(folder_path, '*.gsm'))
        for gsmfile in gsmfiles:
            filename = os.path.splitext(os.path.basename(gsmfile))[0]
            if os.path.isfile(gsmfile):
                if foldername=='--all' or os.path.splitext(os.path.basename(foldername))[0]==filename:
                    if logger:
                        logger.info(f'Processing file: {gsmfile}')
                    print('File:', gsmfile)
                    
                    try:
                        # Entsprechende Aktion ausführen
                        if mode=='l2hsf':
                            gsm2hsf(gsmfile)
                        elif mode=='hsf2l':
                            hsf2gsm(gsmfile)
                        else:
                            error_msg = f'Unknown mode: {mode}'
                            if logger:
                                logger.error(error_msg)
                            print(error_msg)
                            sys.exit(1)
                        if logger:
                            logger.info(f'Successfully processed: {gsmfile}')
                    except Exception as e:
                        error_msg = f'Error processing {gsmfile}: {str(e)}'
                        if logger:
                            logger.error(error_msg, exc_info=True)
                        print(error_msg)
                        # Bei Batch-Verarbeitung weiter machen, bei Einzeldatei beenden
                        if foldername != '--all':
                            sys.exit(1)
